/**
 * \file: sysmount_helper.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: startup
 *
 * \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
 *
 * \copyright (c) 2010, 2011 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/
#include "sysmount_helper.h"
#include "mountdb_defaults.h"
#include "mounter.h"

#include <stdio.h>
#include <assert.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

typedef enum mount_type_t
{
	MOUNT_DEFAULT_COMPLETE,
	MOUNT_DEFAULT_PARTLY,
	MOUNT_FROM_CONF_FILE
} mount_type_t;

static void sysmount_helper_print_help(void);
static error_code_t sysmount_helper_mount_defaults_complete(void);
static error_code_t sysmount_helper_mount_defaults_partly(void);
static error_code_t sysmount_helper_mount_from_file(void);
static error_code_t sysmount_helper_mount_default_mp(char *default_mp);

static mount_type_t mount_type=MOUNT_DEFAULT_COMPLETE;
static char **my_argv=NULL;
static int my_argc=-1;

error_code_t sysmount_helper_init(int argc, char *argv[])
{
	//no arguments, lets mount the complete default set
	if (argc<=1)
		return RESULT_OK;

	if (strcmp(argv[1],"--help")==0 || strcmp(argv[1],"-h")==0)
	{
		sysmount_helper_print_help();
		return RESULT_HELP_PRINTED;
	}

	my_argc=argc;
	my_argv=argv;

	if (strcmp(argv[1],"-c")==0 || strcmp(argv[1],"--conf")==0)
		mount_type=MOUNT_FROM_CONF_FILE;
	else
		mount_type=MOUNT_DEFAULT_PARTLY;

	return RESULT_OK;
}

error_code_t sysmount_helper_start(void)
{
	switch(mount_type)
	{
	case MOUNT_DEFAULT_COMPLETE:
		return sysmount_helper_mount_defaults_complete();
	case MOUNT_DEFAULT_PARTLY:
		return sysmount_helper_mount_defaults_partly();
	case MOUNT_FROM_CONF_FILE:
		return sysmount_helper_mount_from_file();
	default:
		assert(false);
	}
}

void sysmount_helper_deinit(void)
{

}

static void sysmount_helper_print_help(void)
{
	const mount_point_t *default_mp;
	printf("\nUsage: sysmount-helper [<subset of compiled in default mount points>]\n");
	printf("or:    sysmount-helper --conf <conf file>\n");
	printf("or:    sysmount-helper --help\n\n");
	printf("This command mounts a set of system mount points. All system mount points\n");
	printf("compiled in are mounted when sysmount-helper is called without any further\n");
	printf("arguments.\n");
	printf("Following mount points are compiled in:\n");
	default_mp=default_mp_configurations;
	while (default_mp->mp!=NULL)
	{
		printf("-> %s\n", default_mp->mp);
		default_mp++;
	}
	printf("A subset of mount points can be mounted by passing them as arguments to\n");
	printf("sysmount-helper.\n\n");
	printf("A free set of mount points including mount options can be defined in a\n");
	printf("configuration file and passed with --conf option. No default mount points\n");
	printf("are mounted in this case. Each line in the file represents one mount point.\n");
	printf("Following format is expected:\n");
	printf("<mount point>:<source>:<file system>:<options>\n\n");
}

static error_code_t sysmount_helper_mount_defaults_complete()
{
	error_code_t result=RESULT_OK;
	const mount_point_t *default_mp;

	default_mp=default_mp_configurations;
	while (default_mp->mp!=NULL && result==RESULT_OK)
	{
		result=mounter_do_mount(default_mp);
		default_mp++;
	}

	return result;
}

static error_code_t sysmount_helper_mount_defaults_partly()
{
	int i;
	error_code_t result=RESULT_OK;

	//any parameter after the actual command is expected to be a mount point
	for (i=1;i<my_argc && result==RESULT_OK;i++)
		result=sysmount_helper_mount_default_mp(my_argv[i]);

	return result;
}

static error_code_t sysmount_helper_mount_default_mp(char *default_mp)
{
	//we use RESULT_USAGE error code in case we do not find a default mount point for given parameter
	//we set this initially and change it once we found the mount point are trying to mount it
	error_code_t result=RESULT_USAGE;
	const mount_point_t *default_mp_itr;
	size_t len;

	//does mp end with '/'? -> remove it
	len=strlen(default_mp);
	if (default_mp[len-1]=='/')
		default_mp[len-1]='\0';

	default_mp_itr=default_mp_configurations;
	while (default_mp_itr->mp!=NULL && result==RESULT_USAGE)
	{
		if (strcmp(default_mp_itr->mp,default_mp)==0)
			result=mounter_do_mount(default_mp_itr);
		default_mp_itr++;
	}

	if (result==RESULT_USAGE)
	{
		fprintf(stderr, "\"%s\" is not a default mount point. Define this mount point in a\n"
				"configuration file or add it to the list of default mount points and recompile\n"
				"the tool.\n", default_mp);
		sysmount_helper_print_help();
	}

	return result;
}

static error_code_t sysmount_helper_mount_from_file()
{
	error_code_t result=RESULT_OK;
	FILE *conf_file;
	char *line_buf=NULL;
	size_t line_buf_size=0;
	int line_no=0;

	//    0            1       2
	//sysmount-helper -c <conffile>
	if (my_argc < 3)
	{
		fprintf(stderr, "Configuration file missing after parameter --conf.\n");
		sysmount_helper_print_help();
		return RESULT_USAGE;
	}

	conf_file=fopen(my_argv[2],"r");
	if (conf_file == NULL)
	{
		fprintf(stderr, "Unable to read configuration file \"%s\": %s.\n",my_argv[2],strerror(errno));
		return RESULT_CANT_OPEN_CONF_FILE;
	}

	while (getline(&line_buf,&line_buf_size,conf_file)!=-1 && result==RESULT_OK)
	{
		char *trimmed_line=line_buf;
		line_no++;
		//trim line to the left
		while (*trimmed_line==' ' || *trimmed_line=='\t' || *trimmed_line=='\n')
			trimmed_line++;
		//comment or empty?
		if (*trimmed_line=='#' || *trimmed_line=='\0')
			continue;
		result=mounter_do_parse_and_mount(line_no,trimmed_line);
	}

	fclose(conf_file);

	if (line_buf!=NULL)
		free(line_buf);

	return result;
}
